package tonematrix.gui;
import tonematrix.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

/* Component that draws a ToneMatrix, runs the audio loop, and responds to
 * mouse events.
 */
public class ToneMatrixDisplay extends JPanel {
    private ToneMatrix matrix; // Actual tone matrix
    private int totalGridSize; // Its size in pixels
    
    /* How many samples to generate per fill. */
    private static final int SAMPLES_PER_FILL = 400;
    
    /* Border padding size; used to make things Nice and Pretty. */
    private static final int BORDER_SIZE = 20;
    
    /* Coordinate transforms: global-to-local (component x/y to
     * local ToneMatrix x/y) and vice-versa.
     */
     
    /**
     * Given a screen x coordinate, converts it to a local Tone Matrix
     * x coordinate.
     *
     * @param The global x coordinate.
     * @return The Tone Matrix local x coordinate.
     */
    private int globalToLocalX(int x) {
        int result = x - BORDER_SIZE;
        if (result < 0) result = 0;
        if (result >= totalGridSize) result = totalGridSize - 1;
        return result;
    }
    
    /**
     * Given a screen y coordinate, converts it to a local Tone Matrix
     * y coordinate.
     *
     * @param The global y coordinate.
     * @return The Tone Matrix local y coordinate.
     */
    private int globalToLocalY(int y) {
        int result = y - BORDER_SIZE;
        if (result < 0) result = 0;
        if (result >= totalGridSize) result = totalGridSize - 1;
        return result;
    }
    
    /**
     * Given a Tone Matrix x coordinate, converts it to a global screen
     * x coordinate.
     *
     * @param The Tone Matrix local x coordinate.
     * @return The global x coordinate.
     */
    private int localToGlobalX(int x) {
        return x + BORDER_SIZE;    
    }
    
    /**
     * Given a Tone Matrix y coordinate, converts it to a global screen
     * y coordinate.
     *
     * @param The Tone Matrix local y coordinate.
     * @return The global y coordinate.
     */
    private int localToGlobalY(int y) {
        return y + BORDER_SIZE;
    }

    /* Mouse listener type to forward events to the ToneMatrix. */
    private class MouseHandler implements MouseListener, MouseMotionListener {        
        @Override public void mouseClicked(MouseEvent e) {
            // Not forwarded
        }
        
        @Override public void mouseEntered(MouseEvent e) {
            // Not forwarded
        }
        
        @Override public void mouseExited(MouseEvent e) {
            // Not forwarded
        }
        
        @Override public void mousePressed(MouseEvent e) {
            matrix.mousePressed(globalToLocalX(e.getX()), globalToLocalY(e.getY()));
            repaint();
        }
        
        @Override public void mouseReleased(MouseEvent e) {
            // Not forwarded
        }
        
        @Override public void mouseMoved(MouseEvent e) {
            // Not forwarded
        }
        
        @Override public void mouseDragged(MouseEvent e) {
            matrix.mouseDragged(globalToLocalX(e.getX()), globalToLocalY(e.getY()));
            repaint();
        }
    }
    
    /**
     * The magic callback that plays audio. Asks the ToneMatrix for a bunch of
     * samples and sends them to the audio system. To ensure continuous music
     * generation, this function then triggers itself again in the future.
     */
    private void fillBuffer() {
        for (int i = 0; i < SAMPLES_PER_FILL; i++) {
            AudioSystem.play(matrix.nextSample());
        }
        
        /* Keep the party going! */
        SwingUtilities.invokeLater(() -> fillBuffer());
    }
    
    /**
     * Creates a new ToneMatrixDisplay using the given grid size (e.g. 16x16 grid of
     * lights) and size in pixels per light.
     *
     * @param gridSize The dimensions of the Tone Matrix (e.g. 16x16).
     * @param lightSize The size, in pixels, of each light.
     */
    public ToneMatrixDisplay(int gridSize, int lightSize) {
        matrix = new ToneMatrix(gridSize, lightSize);
        totalGridSize = gridSize * lightSize;
        
        setPreferredSize(new Dimension(totalGridSize + 2 * BORDER_SIZE, totalGridSize + 2 * BORDER_SIZE));
    
        /* React to the mouse. */
        MouseHandler handler = new MouseHandler();
        addMouseListener(handler);
        addMouseMotionListener(handler);
        
        /* Launch the audio. Calling this function will initiate a chain
         * of repeated calls.
         */
        fillBuffer();
    }
    
    /**
     * Draws the Tone Matrix on the screen.
     *
     * @param g The graphics object to use for rendering.
     */
    @Override public void paint(Graphics g) {    
        /* Wipe everything that's already there. */
        g.setColor(Color.BLACK);
        g.fillRect(0, 0, getWidth(), getHeight());

        /* Draw the Tone Matrix. */
        matrix.draw(new ActualGraphics(g));
                   
        /* Sync with the OS so that animations look smooth. */
        Toolkit.getDefaultToolkit().sync();
    }
    
    /* The actual graphics system used by ToneMatrix. This converts everything from
     * scene coordinates to screen coordinates before drawing so that everything is
     * always centered.
     */
    private class ActualGraphics implements ToneMatrixGraphics {
        private Graphics graphics;
                
        public ActualGraphics(Graphics g) {
            graphics = g;
        }

        @Override public void drawRectangle(int x, int y, int width, int height, Color color) {
            x = localToGlobalX(x);
            y = localToGlobalY(y);
        
            /* Fill the rectangle and draw a border using a darker color. */
            graphics.setColor(color);
            graphics.fillRect(x, y, width, height);
            graphics.setColor(color.darker());
            graphics.drawRect(x, y, width, height);
        }
    }
}
