package tonematrix;
import java.util.*;
import java.awt.Color;
import testing.*;

/* Test cases for your ToneMatrix type. Feel free to add additional test cases here
 * to exercise edge cases not covered by the provided tests.
 */
public class ToneMatrixTests {
    /* Utility type representing a rectangle that was drawn to the screen. This is used
     * by the testing code so that we can call draw and see what rectangles you drew
     */
    private static final class DrawnRectangle {
        public int x, y, width, height;
        public Color color;
            
        public DrawnRectangle(int x, int y, int width, int height, Color color) {
            this.x      = x;
            this.y      = y;
            this.width  = width;
            this.height = height;
            this.color  = color;
        }
        
        @Override public String toString() {
            return "(" + x + ", " + y + ", " + width + ", " + height + ", " + color + ")";
        }
        
        @Override public boolean equals(Object rhsObj) {
            if (!(rhsObj instanceof DrawnRectangle)) {
                return false;
            }
            
            var rhs = (DrawnRectangle) rhsObj;
            return color.equals(rhs.color) &&
                   x == rhs.x &&
                   y == rhs.y && 
                   width == rhs.width &&
                   height == rhs.height;
        }
        
        @Override public int hashCode() {
            int result = color.hashCode();
            result = 37 * result + x;
            result = 37 * result + y;
            result = 37 * result + width;
            result = 37 * result + height;
            return result;
        }
    }

    /* Utility type that acts like a ToneMatrixGraphics and writes down all the rectangles
     * drawn so that they can be inspected later.
     */
    private static class RectangleCatcher implements ToneMatrixGraphics {        
        /* List of particles drawn thus far. */
        private List<DrawnRectangle> drawn = new ArrayList<>();
        
        public void drawRectangle(int x, int y, int width, int height, Color color) {
            drawn.add(new DrawnRectangle(x, y, width, height, color));
        }
        
        public int numDrawn() {
            return drawn.size();
        }
        
        public DrawnRectangle get(int index) {
            return drawn.get(index);
        }
        
        public void reset() {
            drawn.clear();
        }
    }

    public static void main(String[] args) {
        new TestDriver() {
            @TestCase(milestone = 1, name = "ToneMatrix constructor stores the light dimensions.") public void test1() {
                /* Other tests may have changed the sample rate. This is necessary to ensure that
                 * the sample rate is set to a value large enough for all StringInstruments can
                 * behave correctly.
                 */
                AudioSystem.setSampleRate(44100);

                var matrix1 = new ToneMatrix(16, 137);
                EXPECT_NOT_EQUAL(matrix1.instruments, null);
                EXPECT_EQUAL(matrix1.instruments.length, 16);
                EXPECT_EQUAL(matrix1.lightSize, 137);

                var matrix2 = new ToneMatrix(5, 106);
                EXPECT_NOT_EQUAL(matrix2.instruments, null);
                EXPECT_EQUAL(matrix2.instruments.length, 5);
                EXPECT_EQUAL(matrix2.lightSize, 106);
            }

            @TestCase(milestone = 1, name = "ToneMatrix constructor sets instrument frequencies.") public void test2() {
                AudioSystem.setSampleRate(44100);

                var matrix = new ToneMatrix(16, 137);
                EXPECT_NOT_EQUAL(matrix.instruments, null);

                /* Check that the frequencies are right by computing what they should be and comparing
                 * against the expected value.
                 */
                for (int i = 0; i < 16; i++) {
                    EXPECT_EQUAL(matrix.instruments[i].waveform.length, (int)(AudioSystem.sampleRate() / ToneMatrix.frequencyForRow(i)));
                }
            }

            @TestCase(milestone = 1, name = "ToneMatrix constructor initializes lights to off.") public void test3() {
                AudioSystem.setSampleRate(44100);

                var matrix = new ToneMatrix(13, 137);
                EXPECT_NOT_EQUAL(matrix.grid, null);

                for (int i = 0; i < 13 * 13; i++) {
                    EXPECT_EQUAL(matrix.grid[i], false);
                }
            }

            @TestCase(milestone = 1, name = "mousePressed toggles the light at row 0, col 0.") public void test4() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(14, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_EQUAL(matrix.instruments.length, 14);

                /* Light should be off. */
                EXPECT_EQUAL(matrix.grid[0], false);

                /* Pressing at position (1, 1) presses the upper-left corner. It's off, so
                 * this should turn it on.
                 */
                matrix.mousePressed(1, 1);
                EXPECT_EQUAL(matrix.grid[0], true);

                /* Make sure that every other light is still off. */
                for (int row = 0; row < 14; row++) {
                    for (int col = 0; col < 14; col++) {
                        /* Skip (0, 0) */
                        if (row != 0 || col != 0) {
                            EXPECT_EQUAL(matrix.grid[14 * row + col], false);
                        }
                    }
                }

                /* Do this again, which should turn the light back off. */
                matrix.mousePressed(1, 1);
                EXPECT_EQUAL(matrix.grid[0], false);

                /* Make sure that every other light is still off. */
                for (int row = 0; row < 14; row++) {
                    for (int col = 0; col < 14; col++) {
                        /* Skip (0, 0) */
                        if (row != 0 || col != 0) {
                            EXPECT_EQUAL(matrix.grid[14 * row + col], false);
                        }
                    }
                }
            }

            @TestCase(milestone = 1, name = "mousePressed toggles the light at row 9, col 6.") public void test5() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(10, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);

                final int rowIndex = 9;
                final int colIndex = 6;
                final int lightIndex = 10 * rowIndex + colIndex;

                /* Dead center inside the light. */
                final int lightX = 1 + 2 * colIndex;
                final int lightY = 1 + 2 * rowIndex;

                /* Light should be off. */
                EXPECT_EQUAL(matrix.grid[lightIndex], false);

                /* Pressing at position (lightX, lightY) presses the light. It's off, so
                 * this should turn it on.
                 */
                matrix.mousePressed(lightX, lightY);
                EXPECT_EQUAL(matrix.grid[lightIndex], true);

                /* Make sure that every other light is still off. */
                for (int row = 0; row < 10; row++) {
                    for (int col = 0; col < 10; col++) {
                        /* Skip (rowIndex, colIndex) */
                        if (row != rowIndex || col != colIndex) {
                            EXPECT_EQUAL(matrix.grid[10 * row + col], false);
                        }
                    }
                }

                /* Do this again, which should turn the light back off. */
                matrix.mousePressed(lightX, lightY);
                EXPECT_EQUAL(matrix.grid[lightIndex], false);

                /* Make sure that every other light is still off. */
                for (int row = 0; row < 10; row++) {
                    for (int col = 0; col < 10; col++) {
                        /* Skip (rowIndex, colIndex) */
                        if (row != rowIndex || col != colIndex) {
                            EXPECT_EQUAL(matrix.grid[10 * row + col], false);
                        }
                    }
                }
            }

            @TestCase(milestone = 1, name = "mousePressed works across the top row.") public void test6() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);

                /* Turn all the lights in the top row on. */
                for (int col = 0; col < 16; col++) {
                    /* This is the light at index 0 * 16 + col = col within the grid. */
                    final int index = col;

                    /* This light should be off. */
                    EXPECT_EQUAL(matrix.grid[index], false);

                    /* Press at x = 2*col + 1, y = 1 to press the light. */
                    matrix.mousePressed(2 * col + 1, 1);
                    EXPECT_EQUAL(matrix.grid[index], true);
                }

                /* Make sure everything in the top row is still turned on. */
                for (int col = 0; col < 16; col++) {
                    EXPECT_EQUAL(matrix.grid[col], true);
                }

                /* Make sure everything not in row 0 is turned off. */
                for (int row = 1; row < 16; row++) {
                    for (int col = 0; col < 16; col++) {
                        EXPECT_EQUAL(matrix.grid[16 * row + col], false);
                    }
                }

                /* Turn all the lights in the top row back off. */
                for (int col = 0; col < 16; col++) {
                    /* This is the light at index 0 * 16 + col = col within the grid. */
                    final int index = col;

                    /* This light should be on. */
                    EXPECT_EQUAL(matrix.grid[index], true);

                    /* Press at x = 2*col + 1, y = 1 to press the light. */
                    matrix.mousePressed(2 * col + 1, 1);
                    EXPECT_EQUAL(matrix.grid[index], false);
                }

                /* Make sure all lights are off. */
                for (int row = 0; row < 16; row++) {
                    for (int col = 0; col < 16; col++) {
                        EXPECT_EQUAL(matrix.grid[16 * row + col], false);
                    }
                }
            }

            @TestCase(milestone = 1, name = "mousePressed works across the leftmost column.") public void test7() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);

                /* Turn all the lights in the leftmost column on. */
                for (int row = 0; row < 16; row++) {
                    /* This is the light at index 16 * row + col = 16 * row within the grid. */
                    final int index = 16 * row;

                    /* This light should be off. */
                    EXPECT_EQUAL(matrix.grid[index], false);

                    /* Press at x = 1, y = 2*row + 1 to press the light. */
                    matrix.mousePressed(1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[index], true);
                }

                /* Make sure everything in the leftmost column is still turned on. */
                for (int row = 0; row < 16; row++) {
                    EXPECT_EQUAL(matrix.grid[16 * row], true);
                }

                /* Make sure everything not in col 0 is turned off. */
                for (int row = 0; row < 16; row++) {
                    for (int col = 1; col < 16; col++) {
                        EXPECT_EQUAL(matrix.grid[16 * row + col], false);
                    }
                }

                /* Turn all the lights in the leftmost column back off. */
                for (int row = 0; row < 16; row++) {
                    /* This is the light at index 16 * row + col = 16 * row within the grid. */
                    final int index = 16 * row;

                    /* This light should be on. */
                    EXPECT_EQUAL(matrix.grid[index], true);

                    /* Press at x = 1, y = 2*row + 1 to press the light. */
                    matrix.mousePressed(1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[index], false);
                }

                /* Make sure all lights are off. */
                for (int row = 0; row < 16; row++) {
                    for (int col = 0; col < 16; col++) {
                        EXPECT_EQUAL(matrix.grid[16 * row + col], false);
                    }
                }
            }

            @TestCase(milestone = 2, name = "mouseDragged turns on all lights in the top row.") public void test8() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);

                /* Press the mouse at (1, 1) to turn on the top-left light. */
                EXPECT_EQUAL(matrix.grid[0], false);
                matrix.mousePressed(1, 1);
                EXPECT_EQUAL(matrix.grid[0], true);

                /* Drag the mouse to (3, 1) to turn on the light at row 0, column 1. */
                EXPECT_EQUAL(matrix.grid[1], false);
                matrix.mouseDragged(3, 1);
                EXPECT_EQUAL(matrix.grid[1], true);

                /* Drag the mouse back to (1, 1). This should not have any effect because
                 * the light is already on.
                 */
                EXPECT_EQUAL(matrix.grid[0], true);
                matrix.mouseDragged(1, 1);
                EXPECT_EQUAL(matrix.grid[0], true);

                /* Drag the mouse back to (3, 1). This should not have any effect because
                 * the light is already on.
                 */
                EXPECT_EQUAL(matrix.grid[1], true);
                matrix.mouseDragged(3, 1);
                EXPECT_EQUAL(matrix.grid[1], true);

                /* Drag the mouse across the rest of the top row to turn on all lights. */
                for (int col = 2; col < 16; col++) {
                    /* This is the light at index 0 * 16 + col = col within the grid. */
                    final int index = col;

                    /* This light should be off. */
                    EXPECT_EQUAL(matrix.grid[index], false);

                    /* Press at x = 2*col + 1, y = 1 to press the light. */
                    matrix.mouseDragged(2 * col + 1, 1);
                    EXPECT_EQUAL(matrix.grid[index], true);
                }

                /* Make sure everything in the top row is still turned on. */
                for (int col = 0; col < 16; col++) {
                    EXPECT_EQUAL(matrix.grid[col], true);
                }

                /* Make sure everything not in row 0 is turned off. */
                for (int row = 1; row < 16; row++) {
                    for (int col = 0; col < 16; col++) {
                        EXPECT_EQUAL(matrix.grid[16 * row + col], false);
                    }
                }

                /* Turn all the lights in the top row back off. Begin by pressing the
                 * mouse at row 0, column 15.
                 */
                EXPECT_EQUAL(matrix.grid[15], true);
                matrix.mousePressed(2 * 15 + 1, 1);
                EXPECT_EQUAL(matrix.grid[15], false);

                /* Drag the mouse to row 0, column 14, to turn that light off. */
                EXPECT_EQUAL(matrix.grid[14], true);
                matrix.mouseDragged(2 * 14 + 1, 1);
                EXPECT_EQUAL(matrix.grid[14], false);

                /* Drag the mouse back to row 0, column 15, which should have no
                 * effect because the light is already off.
                 */
                EXPECT_EQUAL(matrix.grid[15], false);
                matrix.mouseDragged(2 * 15 + 1, 1);
                EXPECT_EQUAL(matrix.grid[15], false);

                /* Drag the mouse back to row 0, column 14, which should have no
                 * effect because the light is already off.
                 */
                EXPECT_EQUAL(matrix.grid[14], false);
                matrix.mouseDragged(2 * 14 + 1, 1);
                EXPECT_EQUAL(matrix.grid[14], false);

                /* Now drag from right to left back across the row to turn all the
                 * lights off.
                 */
                for (int col = 13; col >= 0; col--) {
                    /* This is the light at index 0 * 16 + col = col within the grid. */
                    final int index = col;

                    /* This light should be on. */
                    EXPECT_EQUAL(matrix.grid[index], true);

                    /* Press at x = 2*col + 1, y = 1 to press the light. */
                    matrix.mousePressed(2 * col + 1, 1);
                    EXPECT_EQUAL(matrix.grid[index], false);
                }

                /* Make sure all lights are off. */
                for (int row = 0; row < 16; row++) {
                    for (int col = 0; col < 16; col++) {
                        EXPECT_EQUAL(matrix.grid[16 * row + col], false);
                    }
                }
            }

            @TestCase(milestone = 3, name = "Draws a grid where all lights are off.") public void test9() {
                AudioSystem.setSampleRate(44100);

                /* Each cell has size 137 * 137 */
                final int lightSize = 137;
                var matrix = new ToneMatrix(16, lightSize);

                EXPECT_NOT_EQUAL(matrix.grid, null);

                /* RectangleCatcher is a type that captures all rectangles drawn by
                 * drawRectangle rather than rendering them to the screen. We'll use
                 * this to determine which rectangles are drawn.
                 */
                var catcher = new RectangleCatcher();
                matrix.draw(catcher);

                /* Scan the rectangles. Confirm that...
                 *
                 * 1. We didn't get any duplicates.
                 * 2. Every rectangle's coordinates are a multiple of lightSize.
                 * 3. Every rectangle upper-corner is within the box [0, 0] x [lightSize * 15, lightSize * 15].
                 * 4. Every rectangle has width and height equal to lightSize.
                 * 5. Every light is given the OFF color.
                 */
                var drawn = new HashSet<DrawnRectangle>();
                for (int i = 0; i < catcher.numDrawn(); i++) {
                    /* Check for duplicates. */
                    if (drawn.contains(catcher.get(i))) {
                        SHOW_ERROR("Duplicate rectangle drawn: " + catcher.get(i));
                    }
                    drawn.add(catcher.get(i));

                    /* Check that x and y coordinates are multiples of the light size. */
                    var rectangle = catcher.get(i);
                    EXPECT_EQUAL(rectangle.x % lightSize, 0);
                    EXPECT_EQUAL(rectangle.y % lightSize, 0);

                    /* Check that everything is within the appropriate bounding box. */
                    EXPECT_GREATER_THAN_OR_EQUAL_TO(rectangle.x, 0);
                    EXPECT_GREATER_THAN_OR_EQUAL_TO(rectangle.y, 0);
                    EXPECT_LESS_THAN(rectangle.x, 16 * lightSize);
                    EXPECT_LESS_THAN(rectangle.y, 16 * lightSize);

                    /* Check the width and height of the rectangles. */
                    EXPECT_EQUAL(rectangle.width,  lightSize);
                    EXPECT_EQUAL(rectangle.height, lightSize);

                    /* Make sure the light is off. */
                    var color = catcher.get(i).color;
                    EXPECT_EQUAL(color, ToneMatrix.LIGHT_OFF_COLOR);
                }

                /* If there are 256 total rectangles, the above finalraints ensure that
                 * every possible rectangle has been drawn and they've all been drawn
                 * once.
                 */
                EXPECT_EQUAL(catcher.numDrawn(), 16 * 16);
            }

            @TestCase(milestone = 3, name = "Draws a grid where all lights are on.") public void test10() {
                AudioSystem.setSampleRate(44100);

                /* Each cell has size 137 * 137 */
                final int lightSize = 137;
                var matrix = new ToneMatrix(16, lightSize);

                EXPECT_NOT_EQUAL(matrix.grid, null);

                /* Turn all the lights on. To do so, press at position (137 * col + 1, 137 * row + 1)
                 * for all rows and columns.
                 */
                for (int row = 0; row < 16; row++) {
                    for (int col = 0; col < 16; col++) {
                        matrix.mousePressed(lightSize * col + 1, lightSize * row + 1);
                        EXPECT_EQUAL(matrix.grid[16 * row + col], true);
                    }
                }


                /* RectangleCatcher is a type that captures all rectangles drawn by
                 * drawRectangle rather than rendering them to the screen. We'll use
                 * this to determine which rectangles are drawn.
                 */
                var catcher = new RectangleCatcher();
                matrix.draw(catcher);

                /* Scan the rectangles. Confirm that...
                 *
                 * 1. We didn't get any duplicates.
                 * 2. Every rectangle's coordinates are a multiple of lightSize.
                 * 3. Every rectangle upper-corner is within the box [0, 0] x [lightSize * 15, lightSize * 15].
                 * 4. Every rectangle has width and height equal to lightSize.
                 * 5. Every light is given the ON color.
                 */
                var drawn = new HashSet<DrawnRectangle>();
                for (int i = 0; i < catcher.numDrawn(); i++) {
                    /* Check for duplicates. */
                    if (drawn.contains(catcher.get(i))) {
                        SHOW_ERROR("Duplicate rectangle drawn: " + catcher.get(i));
                    }
                    drawn.add(catcher.get(i));

                    /* Check that x and y coordinates are multiples of the light size. */
                    var rectangle = catcher.get(i);
                    EXPECT_EQUAL(rectangle.x % lightSize, 0);
                    EXPECT_EQUAL(rectangle.y % lightSize, 0);

                    /* Check that everything is within the appropriate bounding box. */
                    EXPECT_GREATER_THAN_OR_EQUAL_TO(rectangle.x, 0);
                    EXPECT_GREATER_THAN_OR_EQUAL_TO(rectangle.y, 0);
                    EXPECT_LESS_THAN(rectangle.x, 16 * lightSize);
                    EXPECT_LESS_THAN(rectangle.y, 16 * lightSize);

                    /* Check the width and height of the rectangles. */
                    EXPECT_EQUAL(rectangle.width,  lightSize);
                    EXPECT_EQUAL(rectangle.height, lightSize);

                    /* Make sure the light is off. */
                    var color = catcher.get(i).color;
                    EXPECT_EQUAL(color, ToneMatrix.LIGHT_ON_COLOR);
                }

                /* If there are 256 total rectangles, the above finalraints ensure that
                 * every possible rectangle has been drawn and they've all been drawn
                 * once.
                 */
                EXPECT_EQUAL(catcher.numDrawn(), 16 * 16);
            }

            @TestCase(milestone = 3, name = "Draws a grid with one offset light that's on.") public void test11() {
                AudioSystem.setSampleRate(44100);

                /* Each cell has size 137 * 137 */
                final int lightSize = 137;
                var matrix = new ToneMatrix(16, lightSize);

                EXPECT_NOT_EQUAL(matrix.grid, null);

                /* Row 9, column 6. */
                final int rowIndex = 9;
                final int colIndex = 6;
                final int lightIndex = 16 * rowIndex + colIndex;

                /* Dead center inside the light. */
                final int lightX = 1 + lightSize * colIndex;
                final int lightY = 1 + lightSize * rowIndex;

                matrix.mousePressed(lightX, lightY);
                EXPECT_EQUAL(matrix.grid[lightIndex], true);

                /* Set up a RectangleCatcher to catch all drawn rectangles. */
                var catcher = new RectangleCatcher();
                matrix.draw(catcher);

                /* Confirm we drew 256 rectangles. */
                EXPECT_EQUAL(catcher.numDrawn(), 16 * 16);

                /* Store all rectangles with the ON color. */
                var lightsOn = new HashSet<DrawnRectangle>();
                for (int i = 0; i < catcher.numDrawn(); i++) {
                    if (catcher.get(i).color.equals(ToneMatrix.LIGHT_ON_COLOR)) {
                        lightsOn.add(catcher.get(i));
                    }
                }

                /* There should just be one. */
                EXPECT_EQUAL(lightsOn.size(), 1);
                var rect = lightsOn.iterator().next();

                /* Check its width/height. */
                EXPECT_EQUAL(rect.width,  lightSize);
                EXPECT_EQUAL(rect.height, lightSize);

                /* Confirm it's where it should be. */
                EXPECT_EQUAL(rect.x, colIndex * lightSize);
                EXPECT_EQUAL(rect.y, rowIndex * lightSize);
            }

            @TestCase(milestone = 4, name = "First call to nextSample() plucks appropriate strings.") public void test12() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Press the lights in column 0 in all even-numbered rows. */
                for (int row = 0; row < 16; row += 2) {
                    int lightIndex = 16 * row;

                    /* Should be off. */
                    EXPECT_EQUAL(matrix.grid[lightIndex], false);

                    /* Now it's on. */
                    matrix.mousePressed(1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[lightIndex], true);
                }

                /* Confirm that none of the instruments have been plucked by seeing if
                 * the first sound sample in each instrument is 0.
                 */
                for (int i = 0; i < 16; i++) {
                    EXPECT_EQUAL(matrix.instruments[i].cursor, 0);
                    EXPECT_EQUAL(matrix.instruments[i].waveform[0], 0);
                }

                /* Get the next sample from the Tone Matrix. There are eight instruments
                 * active. Each of them, when plucked, returns +0.05 as its sample. We should
                 * therefore have our sample come back as +0.40.
                 */
                EXPECT_EQUAL(matrix.nextSample(), +0.40);

                /* Inspect the even-numbered instruments. Each should have a cursor at
                 * position 1. The sample there should be equal to +0.05.
                 */
                for (int row = 0; row < 16; row += 2) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }

                /* Inspect the odd-numbered instruments. Their cursors should also have
                 * moved forward to position 1, but all the entries should be 0.
                 */
                for (int row = 1; row < 16; row += 2) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[0], 0.0);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], 0.0);
                }
            }

            @TestCase(milestone = 4, name = "All strings are sampled at each step.") public void test13() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Press the lights in column 0 in all even-numbered rows. */
                for (int row = 0; row < 16; row += 2) {
                    int lightIndex = 16 * row;

                    /* Should be off. */
                    EXPECT_EQUAL(matrix.grid[lightIndex], false);

                    /* Now it's on. */
                    matrix.mousePressed(1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[lightIndex], true);
                }

                /* Confirm that none of the instruments have been plucked by seeing if
                 * the first sound sample in each instrument is 0.
                 */
                for (int i = 0; i < 16; i++) {
                    EXPECT_EQUAL(matrix.instruments[i].cursor, 0);
                    EXPECT_EQUAL(matrix.instruments[i].waveform[0], 0);
                }

                /* Get the next sample from the Tone Matrix. There are eight instruments
                 * active. Each of them, when plucked, returns +0.05 as its sample. We should
                 * therefore have our sample come back as +0.40.
                 */
                EXPECT_EQUAL(matrix.nextSample(), +0.40);

                /* Inspect the even-numbered instruments. Each should have a cursor at
                 * position 1. The sample there should be equal to +0.05.
                 */
                for (int row = 0; row < 16; row += 2) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }

                /* Inspect the odd-numbered instruments. Their cursors should also have
                 * moved forward to position 1, but all the entries should be 0.
                 */
                for (int row = 1; row < 16; row += 2) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[0], 0.0);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], 0.0);
                }

                /* Run ten time steps forward, ensuring all the cursors move. */
                for (int i = 2; i < 10; i++) {
                    matrix.nextSample();

                    for (int row = 0; row < 16; row++) {
                        EXPECT_EQUAL(matrix.instruments[row].cursor, i);
                    }
                }
            }

            @TestCase(milestone = 4, name = "The 8192nd call to nextSample() after the first plucks strings.") public void test14() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Press the lights in column 1 in all even-numbered rows. */
                for (int row = 0; row < 16; row += 2) {
                    int lightIndex = 16 * row + 1;

                    /* Should be off. */
                    EXPECT_EQUAL(matrix.grid[lightIndex], false);

                    /* Now it's on. */
                    matrix.mousePressed(3, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[lightIndex], true);
                }

                /* Run time forward for a bit. */
                for (int time = 0; time < 8192; time++) {
                    /* None of the instruments were plucked, so there should be
                     * no sound.
                     */
                    EXPECT_EQUAL(matrix.nextSample(), 0.0);

                    /* All sound samples in the instruments should still be zero
                     * because nothing has been plucked yet.
                     */
                    for (int i = 0; i < 16; i++) {
                        int cursor = matrix.instruments[i].cursor;
                        EXPECT_EQUAL(matrix.instruments[i].waveform[cursor], 0);
                    }
                }

                /* The next call to matrix.nextSample() will move into the first column.
                 * When that happens, we should pluck all even-numbered strings.
                 */

                /* Get the next sample from the Tone Matrix. There are eight instruments
                 * active. Each of them, when plucked, returns +0.05 as its sample. We should
                 * therefore have our sample come back as +0.40.
                 *
                 * If you are failing this test but all the instruments seem to be properly
                 * plucked, make sure that you're resetting the cursor for each instrument
                 * to 0 when calling pluck(). Otherwise, the cursors of the different
                 * instruments might be in arbitrary positions within each waveform and you
                 * may have contributions of +0.05 from some plucked strings and -0.05 from
                 * others.
                 */
                EXPECT_EQUAL(matrix.nextSample(), +0.40);

                /* Inspect the even-numbered instruments. The current sample should
                 * be +0.05.
                 */
                for (int row = 0; row < 16; row += 2) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }

                /* Inspect the odd-numbered instruments. The item under their cursors should
                 * still be 0 because they haven't been plucked yet.
                 */
                for (int row = 1; row < 16; row += 2) {
                    int cursor = matrix.instruments[row].cursor;
                    EXPECT_EQUAL(matrix.instruments[row].waveform[cursor], 0.0);
                }
            }

            @TestCase(milestone = 4, name = "Each instrument is plucked at the appropriate time.") public void test15() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Press the lights all the way down the main diagonal. This will cause each
                 * instrument to be plucked when its column comes up.
                 */
                for (int row = 0; row < 16; row++) {
                    int lightIndex = 16 * row + row;

                    /* Should be off. */
                    EXPECT_EQUAL(matrix.grid[lightIndex], false);

                    /* Now it's on. */
                    matrix.mousePressed(2 * row + 1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[lightIndex], true);
                }

                /* Run the Tone Matrix for full sweep-through. */
                for (int i = 0; i < 16; i++) {
                    /* This call to nextSample() should pluck the instrument
                     * in row i. All previous instruments will have been
                     * plucked before this, and all future instruments will
                     * not have been plucked.
                     */
                    matrix.nextSample();

                    /* Because of the decay rate, none of the previous instrument
                     * amplitudes should be +-0.05. If they were, it means they
                     * were plucked.
                     */
                    for (int before = 0; before < i; before++) {
                        /* 'fabs' is "floating-point absolute value." It's basically
                         * the absolute value function.
                         */
                        int cursor = matrix.instruments[before].cursor;
                        double amplitude = Math.abs(matrix.instruments[before].waveform[cursor]);
                        EXPECT_LESS_THAN(amplitude, +0.05);
                        EXPECT_GREATER_THAN(amplitude, -0.05);
                    }

                    /* Confirm instrument in row i is plucked. */
                    EXPECT_EQUAL(matrix.instruments[i].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[i].waveform[1], +0.05);

                    /* Nothing after us should be plucked. */
                    for (int after = i + 1; after < 16; after++) {
                        int cursor = matrix.instruments[after].cursor;
                        EXPECT_EQUAL(matrix.instruments[after].waveform[cursor], 0.0);
                    }

                    /* Advance time forward 8191 steps. */
                    for (int time = 0; time < 8191; time++) {
                        matrix.nextSample();
                    }
                }
            }

            @TestCase(milestone = 4, name = "nextSample() wraps back around to first column.") public void test16() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Set only the top-left light to on. */
                matrix.mousePressed(1, 1);
                EXPECT_EQUAL(matrix.grid[0], true);

                /* Run several loops through the full Tone Matrix cycle. */
                for (int loopsThrough = 0; loopsThrough < 10; loopsThrough++) {
                    for (int col = 0; col < 16; col++) {
                        /* Column 0 will cause the first string to be plucked. */
                        if (col == 0) {
                            /* Only the first string vibrates, and we just plucked it. */
                            EXPECT_EQUAL(matrix.nextSample(), +0.05);

                            /* First string should have been plucked. */
                            EXPECT_EQUAL(matrix.instruments[0].cursor, 1);
                            EXPECT_EQUAL(matrix.instruments[0].waveform[1], +0.05);
                        }
                        /* Otherwise, nothing was plucked. We can't easily calculate what
                         * the amplitude of the sample is.
                         */
                        else {
                            matrix.nextSample();
                        }

                        /* No other strings should have been plucked. */
                        for (int row = 1; row < 16; row++) {
                            int cursor = matrix.instruments[row].cursor;
                            EXPECT_EQUAL(matrix.instruments[row].waveform[cursor], 0.0);
                        }

                        /* Move through 8191 more samples, which gets to the point where we are
                         * about to pluck things again.
                         */
                        for (int time = 0; time < 8191; time++) {
                            matrix.nextSample();
                        }
                    }
                }
            }

            @TestCase(milestone = 5, name = "resize() chooses instruments of correct frequencies.") public void test17() {
                AudioSystem.setSampleRate(44100);

                /* Initially, a 4x4 grid. */
                var matrix = new ToneMatrix(4, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_EQUAL(matrix.instruments.length, 4);
                EXPECT_EQUAL(matrix.lightSize, 2);

                /* Check the existing frequencies. */
                for (int row = 0; row < 4; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].waveform.length, (int)(AudioSystem.sampleRate() / ToneMatrix.frequencyForRow(row)));
                }

                /* Now expand up to 20 rows. */
                matrix.resize(20);
                EXPECT_EQUAL(matrix.instruments.length, 20);
                EXPECT_EQUAL(matrix.lightSize, 2); // Unchanged

                /* Check the new frequencies. */
                for (int row = 0; row < 20; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].waveform.length, (int)(AudioSystem.sampleRate() / ToneMatrix.frequencyForRow(row)));
                }

                /* Now resize back down to 3 instruments. */
                matrix.resize(3);
                EXPECT_EQUAL(matrix.instruments.length, 3);
                EXPECT_EQUAL(matrix.lightSize, 2); // Unchanged

                /* Check the new frequencies. */
                for (int row = 0; row < 3; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].waveform.length, (int)(AudioSystem.sampleRate() / ToneMatrix.frequencyForRow(row)));
                }
            }

            @TestCase(milestone = 5, name = "resize() preserves existing instruments.") public void test18() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(16, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Press the lights all the way down the first column. This will cause all
                 * instruments to play on the first call to nextSample().
                 */
                for (int row = 0; row < 16; row++) {
                    int lightIndex = 16 * row + 0;

                    /* Should be off. */
                    EXPECT_EQUAL(matrix.grid[lightIndex], false);

                    /* Now it's on. */
                    matrix.mousePressed(1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[lightIndex], true);
                }

                /* Sample the Tone Matrix once to pluck all the instruments. */
                matrix.nextSample();

                /* All instruments should have been plucked, which we can measure
                 * by looking at the underlying waveforms.
                 */
                for (int row = 0; row < 16; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }

                /* Now, resize the matrix down from 16 instruments to 8. This should
                 * preserve the first eight instruments.
                 */
                matrix.resize(8);
                EXPECT_EQUAL(matrix.instruments.length, 8);
                EXPECT_EQUAL(matrix.lightSize,     2);

                /* All instruments should have been plucked, which we can measure
                 * by looking at the underlying waveforms.
                 */
                for (int row = 0; row < 8; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }
            }

            @TestCase(milestone = 5, name = "resize() extends existing instruments with new ones.") public void test19() {
                AudioSystem.setSampleRate(44100);

                /* Make each light 2x2. This makes the light at position 2*col + 1, 2*row + 1 dead
                 * center in the middle of position (row, col).
                 */
                var matrix = new ToneMatrix(8, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);

                /* Press the lights all the way down the first column. This will cause all
                 * instruments to play on the first call to nextSample().
                 */
                for (int row = 0; row < 8; row++) {
                    int lightIndex = 8 * row + 0;

                    /* Should be off. */
                    EXPECT_EQUAL(matrix.grid[lightIndex], false);

                    /* Now it's on. */
                    matrix.mousePressed(1, 2 * row + 1);
                    EXPECT_EQUAL(matrix.grid[lightIndex], true);
                }

                /* Sample the Tone Matrix once to pluck all the instruments. */
                matrix.nextSample();

                /* All instruments should have been plucked, which we can measure
                 * by looking at the underlying waveforms.
                 */
                for (int row = 0; row < 8; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }

                /* Now, resize the matrix up from 8 instruments to 15. This should leave
                 * the first 8 instruments the same and add seven new ones.
                 */
                matrix.resize(15);
                EXPECT_EQUAL(matrix.instruments.length, 15);
                EXPECT_EQUAL(matrix.lightSize,      2);

                /* First eight instruments should remain plucked. */
                for (int row = 0; row < 8; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 1);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[1], +0.05);
                }

                /* Next seven instruments should be unplucked. */
                for (int row = 8; row < 15; row++) {
                    EXPECT_EQUAL(matrix.instruments[row].cursor, 0);
                    EXPECT_EQUAL(matrix.instruments[row].waveform[0], 0.0);
                }
            }

            @TestCase(milestone = 5, name = "resize() preserves old lights when expanding.") public void test20() {
                AudioSystem.setSampleRate(44100);

                var matrix = new ToneMatrix(2, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);
                EXPECT_EQUAL(matrix.instruments.length, 2);

                /* Turn every light in the grid on. The grid should now look like
                 * this:
                 *
                 *    ON ON
                 *    ON ON
                 *
                 */
                for (int i = 0; i < 4; i++) {
                    matrix.grid[i] = true;
                }

                /* Now resize the grid to dimension 3x3. Preserving the lights
                 * should result in a grid that looks like this:
                 *
                 *    ON ON __
                 *    ON ON __
                 *    __ __ __
                 */
                matrix.resize(3);
                EXPECT_EQUAL(matrix.instruments.length, 3);

                EXPECT_EQUAL(matrix.grid[0],  true);
                EXPECT_EQUAL(matrix.grid[1],  true);
                EXPECT_EQUAL(matrix.grid[2], false);
                EXPECT_EQUAL(matrix.grid[3],  true);
                EXPECT_EQUAL(matrix.grid[4],  true);
                EXPECT_EQUAL(matrix.grid[5], false);
                EXPECT_EQUAL(matrix.grid[6], false);
                EXPECT_EQUAL(matrix.grid[7], false);
                EXPECT_EQUAL(matrix.grid[8], false);
            }

            @TestCase(milestone = 5, name = "resize() preserves old lights when contracting.") public void test21() {
                AudioSystem.setSampleRate(44100);

                var matrix = new ToneMatrix(3, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);
                EXPECT_EQUAL(matrix.instruments.length, 3);

                /* Turn every light in the grid on. The grid should now look like
                 * this:
                 *
                 *    ON ON ON
                 *    ON ON ON
                 *    ON ON ON
                 */
                for (int i = 0; i < 9; i++) {
                    matrix.grid[i] = true;
                }

                /* Now resize the grid to dimension 2x2. Preserving the lights
                 * should result in a grid that looks like this:
                 *
                 *    ON ON
                 *    ON ON
                 */
                matrix.resize(2);
                EXPECT_EQUAL(matrix.instruments.length, 2);

                EXPECT_EQUAL(matrix.grid[0],  true);
                EXPECT_EQUAL(matrix.grid[1],  true);
                EXPECT_EQUAL(matrix.grid[2],  true);
                EXPECT_EQUAL(matrix.grid[3],  true);
            }

            @TestCase(milestone = 5, name = "resize() resets the left-to-right sweep.") public void test22() {
                AudioSystem.setSampleRate(44100);

                var matrix = new ToneMatrix(3, 2);
                EXPECT_NOT_EQUAL(matrix.grid, null);
                EXPECT_NOT_EQUAL(matrix.instruments, null);
                EXPECT_NOT_EQUAL(matrix.instruments[0].waveform, null);
                EXPECT_EQUAL(matrix.instruments.length, 3);

                /* Turn on the lights in the second column. */
                matrix.grid[1] = matrix.grid[4] = matrix.grid[7] = true;

                /* Play some sounds for a bit, nothing should be generated. */
                for (int i = 0; i < 1000; i++) {
                    EXPECT_EQUAL(matrix.nextSample(), 0);
                }

                /* Resize down to a 2x2 grid. */
                matrix.resize(2);
                EXPECT_EQUAL(matrix.instruments.length, 2);

                /* Confirm the lights are as follows:
                 *
                 *   __ ON
                 *   __ ON
                 *
                 */
                EXPECT_EQUAL(matrix.grid[0], false);
                EXPECT_EQUAL(matrix.grid[1],  true);
                EXPECT_EQUAL(matrix.grid[2], false);
                EXPECT_EQUAL(matrix.grid[3],  true);

                /* Play 8,192 samples. Nothing should happen. */
                for (int i = 0; i < 8192; i++) {
                    EXPECT_EQUAL(matrix.nextSample(), 0);
                }

                /* The next one, though, should trigger something. */
                double result = matrix.nextSample();
                EXPECT_GREATER_THAN(result, 0);
            }

        }.runTests();   
    }
}
