package tonematrix;
import java.util.*;
import testing.*;

/* Test cases for your StringInstrument type. Feel free to add additional test cases here
 * to exercise edge cases not covered by the provided tests.
 */
public class StringInstrumentTests {  
    public static void main(String[] args) {
        new TestDriver() {
            @TestCase(milestone = 2, name = "Waveform array initialized correctly.") public void test1() {
                /* Change the sample rate to 3, just to make the numbers come out nice. */
                AudioSystem.setSampleRate(3);

                /* Create a string that vibrates at 1hz. This is well below the human hearing
                 * threshold and exists purely for testing purposes.
                 */
                var instrument = new StringInstrument(1);

                /* Make sure something was allocated. */
                EXPECT_NOT_EQUAL(instrument.waveform, null);

                /* Length should be 3 / 1 = 3. */
                EXPECT_EQUAL(instrument.waveform.length, 3);

                /* All entries should be zero. */
                EXPECT_EQUAL(instrument.waveform[0], 0);
                EXPECT_EQUAL(instrument.waveform[1], 0);
                EXPECT_EQUAL(instrument.waveform[2], 0);
            }

            @TestCase(milestone = 2, name = "Constructor reports errors on bad inputs.") public void test2() {
                /* To make the math easier. */
                AudioSystem.setSampleRate(10);

                EXPECT_ERROR(() -> new StringInstrument(-1));  // Negative frequency
                EXPECT_ERROR(() -> new StringInstrument(0));   // Zero frequency
                EXPECT_ERROR(() -> new StringInstrument(10));  // Array would have length 1
                EXPECT_ERROR(() -> new StringInstrument(100)); // Array would have length 0

                /* But we shouldn't get errors for good values. */
                var peachyKeen = new StringInstrument(1);
                EXPECT_NOT_EQUAL(peachyKeen.waveform, null);
            }

            @TestCase(milestone = 2, name = "Constructor sets cursor to position 0.") public void test3() {
                AudioSystem.setSampleRate(10);

                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);

                EXPECT_EQUAL(instrument.cursor, 0);
            }

            @TestCase(milestone = 3, name = "pluck does not allocate a new array.") public void test4() {
                AudioSystem.setSampleRate(10);

                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);

                /* Plucking the string should change the contents of the array, but not
                 * which array we're pointing at.
                 */
                double[] oldArray = instrument.waveform;
                instrument.pluck();

                EXPECT_EQUAL(instrument.waveform, oldArray);
            }

            @TestCase(milestone = 3, name = "pluck sets values to +0.05 and -0.05.") public void test5() {
                AudioSystem.setSampleRate(4);

                /* 4 samples per sec / 1Hz = 4 samples. */
                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);
                EXPECT_EQUAL(instrument.waveform.length, 4);

                instrument.pluck();
                EXPECT_EQUAL(instrument.waveform[0], +0.05);
                EXPECT_EQUAL(instrument.waveform[1], +0.05);
                EXPECT_EQUAL(instrument.waveform[2], -0.05);
                EXPECT_EQUAL(instrument.waveform[3], -0.05);
            }

            @TestCase(milestone = 3, name = "pluck resets the cursor.") public void test6() {
                AudioSystem.setSampleRate(4);

                /* 4 samples per sec / 1Hz = 4 samples. */
                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);

                /* Invasively move the cursor forward. This is called an "invasive"
                 * test because it manipulates internal state of the type we're
                 * testing, rather than just using the interface.
                 */
                instrument.cursor = 3;
                instrument.pluck();
                EXPECT_EQUAL(instrument.cursor, 0);
            }

            @TestCase(milestone = 4, name = "nextSample works if pluck not called.") public void test7() {
                AudioSystem.setSampleRate(10);

                /* 10 samples per sec / 1Hz = 10 samples. */
                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);
                EXPECT_EQUAL(instrument.waveform.length, 10);

                EXPECT_EQUAL(instrument.nextSample(), 0);
                EXPECT_EQUAL(instrument.nextSample(), 0);
                EXPECT_EQUAL(instrument.nextSample(), 0);
                EXPECT_EQUAL(instrument.nextSample(), 0);
                EXPECT_EQUAL(instrument.cursor, 4);
            }

            @TestCase(milestone = 4, name = "nextSample updates waveform array.") public void test8() {
                AudioSystem.setSampleRate(4);

                /* 4 samples per sec / 1Hz = 4 samples. */
                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);
                EXPECT_EQUAL(instrument.waveform.length, 4);

                instrument.pluck();
                EXPECT_EQUAL(instrument.nextSample(), +0.05);
                EXPECT_EQUAL(instrument.nextSample(), +0.05);
                EXPECT_EQUAL(instrument.nextSample(), -0.05);
                EXPECT_EQUAL(instrument.cursor, 3);

                /* The first array value is the average of +0.05 and +0.05, scaled by 0.995.
                 * The two values are the same, so we should get back +0.05 scalled by 0.995.
                 */
                EXPECT_EQUAL(instrument.waveform[0], +0.05 * 0.995);

                /* The next array value is the average of +0.05 and -0.05, scaled by 0.995.
                 * This is exactly zero.
                 */
                EXPECT_EQUAL(instrument.waveform[1], 0.0);

                /* The next array value is the average of -0.05 and -0.05, scaled by 0.995.
                 * As with the first entry, this is -0.05 scaled by 0.995.
                 */
                EXPECT_EQUAL(instrument.waveform[2], -0.05 * 0.995);
            }

            @TestCase(milestone = 4, name = "nextSample wraps around properly.") public void test9() {
                AudioSystem.setSampleRate(2);

                /* 2 samples per sec / 1Hz = 2 samples. */
                var instrument = new StringInstrument(1);
                EXPECT_NOT_EQUAL(instrument.waveform, null);
                EXPECT_EQUAL(instrument.waveform.length, 2);

                /* Pluck the string, forming the array [+0.05, -0.05] */
                instrument.pluck();

                /* Read two samples, which should be +0.05 and -0.05. */
                EXPECT_EQUAL(instrument.nextSample(), +0.05);
                EXPECT_EQUAL(instrument.cursor, 1);
                EXPECT_EQUAL(instrument.nextSample(), -0.05);
                EXPECT_EQUAL(instrument.cursor, 0);

                /* The first array value is the average of +0.05 and -0.05, scaled by 0.995.
                 * This is zero.
                 */
                EXPECT_EQUAL(instrument.waveform[0], 0.0);

                /* The next array value is the average of -0.05 and 0, scaled by 0.995. */
                double decayedTerm = 0.995 * (-0.05 + 0) / 2.0;
                EXPECT_EQUAL(instrument.waveform[1], decayedTerm);

                /* Get two more samples. The waveform is [0, decayedTerm], so we should
                 * get back 0, then decayedTerm.
                 */
                EXPECT_EQUAL(instrument.nextSample(), 0.0);
                EXPECT_EQUAL(instrument.cursor, 1);
                EXPECT_EQUAL(instrument.nextSample(), decayedTerm);
                EXPECT_EQUAL(instrument.cursor, 0);

                /* The first array value is the average of 0.0 and decayedTerm, scaled by
                 * 0.995.
                 */
                double moreDecayed = 0.995 * (decayedTerm + 0) / 2.0;
                EXPECT_EQUAL(instrument.waveform[0], moreDecayed);

                /* The second array value is the average of decayedTerm and moreDecayed,
                 * scaled by 0.995.
                 */
                EXPECT_EQUAL(instrument.waveform[1], 0.995 * (decayedTerm + moreDecayed) / 2.0);
            }

        }.runTests();   
    }
}
