package testing;

import java.lang.reflect.*;
import java.util.*;

/* To use this test driver, inherit from TestDriver and define a main function of
 * the following form:
 *
 *    public static void main(String[] args) {
 *        new MyType().runTests();
 *    }
 *
 * This will locate all instance methods with zero args that are tagged with the
 * @TestCase annotation and run them.
 */
public class TestDriver {
    /**
     * Internal exception type generated by the testing framework. Thrown if
     * an assertion fails.
     */
    private static final class TestFailedException extends RuntimeException {
        public TestFailedException(String message) {
            super(message);
        }
    }
    
    /**
     * Immediately fails a test with a given error message.
     *
     * @param error The error message to display.
     */
    protected void SHOW_ERROR(String error) {
        throw new TestFailedException(error);
    }
    
    /**
     * Checks if the two quantities are equal, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be.
     */
    protected void EXPECT_EQUAL(Object value, Object expected) {
        if (expected == null) {
            if (value != null && !value.equals(expected)) {
                throw new TestFailedException("EXPECT_EQUAL: " + value + " != " + expected);
            }
        } else if (!expected.equals(value)) {
            throw new TestFailedException("EXPECT_EQUAL: " + value + " != " + expected);
        }
    }
    
    /**
     * Checks if the two quantities are equal, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be.
     */
    protected void EXPECT_EQUAL(double value, double expected) {
        if (Math.abs(value - expected) > 1e-8) {
            throw new TestFailedException("EXPECT_EQUAL: " + value + " != " + expected);
        }
    }
    
    /**
     * Checks if the two quantities are equal, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be.
     */
    protected void EXPECT_EQUAL(int value, int expected) {
        if (value != expected) {
            throw new TestFailedException("EXPECT_EQUAL: " + value + " != " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is smaller than the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be smaller than.
     */
    protected void EXPECT_LESS_THAN(int value, int expected) {
        if (value >= expected) {
            throw new TestFailedException("EXPECT_LESS_THAN: " + value + " >= " + expected);
        }
    }

    /**
     * Checks if the first quantity is smaller than the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be smaller than.
     */
    protected void EXPECT_LESS_THAN(double value, double expected) {
        if (value >= expected) {
            throw new TestFailedException("EXPECT_LESS_THAN: " + value + " >= " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is greater than the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be greater than.
     */
    protected void EXPECT_GREATER_THAN(int value, int expected) {
        if (value <= expected) {
            throw new TestFailedException("EXPECT_GREATER_THAN: " + value + " <= " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is greater than the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be greater than.
     */
    protected void EXPECT_GREATER_THAN(double value, double expected) {
        if (value <= expected) {
            throw new TestFailedException("EXPECT_GREATER_THAN: " + value + " <= " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is greater than or equal to the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be not less than.
     */
    protected void EXPECT_GREATER_THAN_OR_EQUAL_TO(int value, int expected) {
        if (value < expected) {
            throw new TestFailedException("EXPECT_GREATER_THAN_OR_EQUAL_TO: " + value + " < " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is greater than or equal to the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should be not less than.
     */
    protected void EXPECT_GREATER_THAN_OR_EQUAL_TO(double value, double expected) {
        if (value < expected) {
            throw new TestFailedException("EXPECT_GREATER_THAN: " + value + " < " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is not equal to the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should not be.
     */
    protected void EXPECT_NOT_EQUAL(Object value, Object expected) {
        if (expected == null) {
            if (value == null || value.equals(expected)) {
                throw new TestFailedException("EXPECT_NOT_EQUAL: " + value + " == " + expected);
            }
        } else if (expected.equals(value)) {
            throw new TestFailedException("EXPECT_NOT_EQUAL: " + value + " == " + expected);
        }
    }
    
    /**
     * Checks if the first quantity is not equal to the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should not be.
     */
    protected void EXPECT_NOT_EQUAL(double value, double expected) {
        if (Math.abs(value - expected) <= 1e-8) {
            throw new TestFailedException("EXPECT_NOT_EQUAL: " + value + " == " + expected);
        }
    }

    /**
     * Checks if the first quantity is not equal to the second, failing if that's not the case.
     *
     * @param value The value in question.
     * @param expected What it should not be.
     */
    protected void EXPECT_NOT_EQUAL(int value, int expected) {
        if (value == expected) {
            throw new TestFailedException("EXPECT_NOT_EQUAL: " + value + " == " + expected);
        }
    }
    
    /**
     * Checks if a condition is true, failing if that's not the case.
     *
     * @param condition The condition that should be true.
     */
    protected void EXPECT(boolean condition) {
        if (!condition) {
            throw new TestFailedException("EXPECT: Condition is not true.");
        }
    }
    
    /**
     * Checks if an error is generated when the given callback is invoked, failing if that's not the case.
     *
     * @param callback The callback to invoke.
     */
    protected void EXPECT_ERROR(TestLambda callback) {
        try {
            callback.invoke();
            throw new TestFailedException("EXPECT_ERROR: Condition did not generate an exception.");
        } catch (Exception e) {
            // Do nothing; all is good.
        }
    }

    /**
     * Runs all tests declared within this type.
     */
    public void runTests() {
        int numTests = 0, numPasses = 0;
        for (Method testCase: findTestCasesIn(getClass())) {
            numTests++;
            
            /* The annotation @TestCase has information about which milestone the test
             * is for and what the name of the test is.
             */
            var data = testCase.getAnnotation(TestCase.class);
            System.out.println("Milestone " + data.milestone() + ": " + data.name());
            
            System.out.print("    Running... ");
            try {
                /* Run the test. If it doesn't fail with an exception, call it a win. */
                testCase.invoke(this);
                System.out.println("pass");
                numPasses++;
            } catch (InvocationTargetException e) {
                /* Generated exception could either be our own TestFailedException, which happens if
                 * EXPECT* fails, or something else.
                 */
                var source = e.getCause();
                if (source instanceof TestFailedException) {
                    System.out.println("fail");
                    System.out.println("       " + source.getMessage());
                    
                    /* Top of the stack trace is the EXPECT* call, so we want one below that. */
                    var where = source.getStackTrace()[1];
                    System.out.println("       " + where.getFileName() + ", Line " + where.getLineNumber());
                } else {
                    /* Other exception. Print the stack trace for debugging info. */
                    System.out.println("exception");
                    System.out.println("        " + source.getClass().getName() + ": " + source.getMessage());
                    for (var elem: e.getCause().getStackTrace()) {
                        System.out.println("            at " + elem);
                    }
                }
            } catch (IllegalArgumentException e) {
                System.out.println("<internal error>");
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                System.out.println("<internal error>");
                e.printStackTrace();
            }
            
            /* Newline to separate out tests. */
            System.out.println();
        }
        
        /* Summarize test performance. */
        if (numTests == numPasses) {
            System.out.println("All tests passed!");
        } else {
            System.out.println(numPasses + " / " + numTests + " test" + (numTests == 1? "" : "s") + " passed.");
        }
    }
    
    /**
     * Locates all test cases defined within a given class. Test cases are defined using the
     * @TestCase annotation on a method.
     *
     * @param clazz The class to search.
     * @return A list of all the test case methods in that class.
     */
    private List<Method> findTestCasesIn(Class<?> clazz) {
        try {
            var result = new ArrayList<Method>();
            
            /* Look at all methods of the class. */
            for (Method method: clazz.getDeclaredMethods()) {       
                /* The method must take no parameters, be annotated as a test case, and
                 * return void to be considered a test case.
                 */  
                if (method.getParameterCount() == 0 &&
                    method.getAnnotation(TestCase.class) != null &&
                    method.getReturnType() == Void.TYPE) {
                        method.setAccessible(true);
                        result.add(method);
                }
            }
            
            /* Sort first by milestone, then by name. Alas, there is no easy way
             * to sort by line number, which is really what we'd like to do here.
             */
            result.sort((Method lhs, Method rhs) -> {
                var one = lhs.getAnnotation(TestCase.class);
                var two = rhs.getAnnotation(TestCase.class);
                
                if (one.milestone() != two.milestone()) {
                    return one.milestone() - two.milestone();
                }
                return one.name().compareTo(two.name());
            });
            
            return result;
        } catch (SecurityException e) {
            throw new RuntimeException("SecurityException when trying to find test cases.", e);
        }
    }
}
